<?php
// تعریف مسیرهای اصلی
define('ROOT_DIR', dirname(dirname(__DIR__)));
define('CONFIG_FILE', ROOT_DIR . '/botdata/testconfig.ini');

// خواندن تنظیمات
if (!file_exists(CONFIG_FILE)) {
    exit('Configuration error');
}

$config = parse_ini_file(CONFIG_FILE, true);
if (!$config) {
    exit('Configuration error');
}

// تنظیمات ربات
$botToken = $config['bot']['token'] ?? '';
$adminId = $config['bot']['admin_id'] ?? '';
$channel = $config['bot']['support_channel'] ?? '';
$channelUrl = $config['bot']['channel_url'] ?? '';
$supportUrl = $config['bot']['support_url'] ?? 'https://t.me/imadmintwo';
$link1 = $config['bot']['link1'] ?? '';
$link2 = $config['bot']['link2'] ?? '';
$link3 = $config['bot']['link3'] ?? '';
$link4 = $config['bot']['link4'] ?? '';
$link5 = $config['bot']['link5'] ?? '';

// تنظیمات دیتابیس
$dbHost = $config['database']['host'] ?? '';
$dbUser = $config['database']['username'] ?? '';
$dbPass = $config['database']['password'] ?? '';
$dbName = $config['database']['dbname'] ?? '';

// تنظیمات مسیر لاگ‌ها
$logDirectory = $config['logs']['directory'] ?? ROOT_DIR . '/public_html/tester';
$botErrorsLog = $logDirectory . '/bot_errors.log';
$errorLog = $logDirectory . '/error.log';

// بررسی تنظیمات ضروری
if (empty($botToken) || empty($dbHost) || empty($dbUser) || empty($dbName) || empty($adminId)) {
    exit('Configuration error');
}

// تنظیم Headerهای امنیتی
header("X-Content-Type-Options: nosniff");
header("X-Frame-Options: DENY");
header("X-XSS-Protection: 1; mode=block");
header("Strict-Transport-Security: max-age=31536000; includeSubDomains");
header("Content-Security-Policy: default-src 'self'");
header("Referrer-Policy: strict-origin-when-cross-origin");
header("Permissions-Policy: geolocation=(), microphone=(), camera=()");

// جلوگیری از caching برای صفحات حساس
if (isset($update['message']['from']['id']) && $update['message']['from']['id'] == $adminId) {
    header("Cache-Control: no-cache, no-store, must-revalidate");
    header("Pragma: no-cache");
    header("Expires: 0");
}

// ایجاد پوشه لاگ‌ها در صورت عدم وجود
if (!file_exists($logDirectory)) {
    mkdir($logDirectory, 0755, true);
}

// تنظیم مسیر لاگ‌های PHP
ini_set('log_errors', 1);
ini_set('error_log', $errorLog);

// بهینه‌سازی: افزایش زمان اجرای اسکریپت برای مدیریت لود بالاتر
ini_set('max_execution_time', 300);

// فعال‌سازی دیباگ و لاگینگ بیشتر برای پیدا کردن مشکلات
error_reporting(E_ALL);
ini_set('display_errors', 1);

// NEW: Batch logging - جمع آوری لاگ‌ها در array و نوشتن در پایان
$botErrorLogs = array();
$generalErrorLogs = array();

// تابع پاکسازی داده‌های لاگ
function sanitizeLogData($data) {
    if (is_array($data)) {
        return json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
    
    $data = str_replace(array("\r", "\n"), array('\\r', '\\n'), $data);
    return substr($data, 0, 1000);
}

// تابع sanitize برای پیام‌های خطا
function sanitizeErrorMessage($message) {
    // موجود: برای کاربر، هاست، پسورد، توکن
    $message = preg_replace("/user '.*?'/i", "user '[REDACTED]'", $message);
    $message = preg_replace("/@'.*?'/i", "@'[REDACTED]'", $message);
    $message = preg_replace("/(host|hostname|server|username):?\s*[\'\"]?[^\'\" ]+[\'\"]?/i", "$1: [REDACTED]", $message);
    $message = preg_replace("/password:?\s*[\'\"]?[^\'\" ]+[\'\"]?/i", "password: [REDACTED]", $message);
    $message = preg_replace("/token:?\s*[\'\"]?[^\'\" ]+[\'\"]?/i", "token: [REDACTED]", $message);
    $message = preg_replace("/bot[0-9a-zA-Z:_-]+/i", "bot[REDACTED]", $message);
    $message = preg_replace("/\[(tcp|unix):\/\/.*?\]/i", "[REDACTED]", $message);
    $message = preg_replace("/(url|path|file):?\s*[\'\"]?[^\'\" ]+[\'\"]?/i", "$1: [REDACTED]", $message);
    $message = preg_replace("/user_id:?\s*\d+/i", "user_id: [REDACTED]", $message);

    // جدید: برای IPها و URLها
    // IPv4 (e.g., 192.168.1.1)
    $message = preg_replace('/\b(?:\d{1,3}\.){3}\d{1,3}\b/', '[REDACTED IP]', $message);
    // IPv6 (e.g., 2001:0db8:85a3:0000:0000:8a2e:0370:7334)
    $message = preg_replace('/\b(?:[0-9a-fA-F]{1,4}:){7}[0-9a-fA-F]{1,4}\b/', '[REDACTED IPv6]', $message);
    // URLها (e.g., https://example.com/path?query=1)
    $message = preg_replace('/\b(https?|ftp|file|http):\/\/[-A-Za-z0-9+&@#\/%?=~_|!:,.;]*[-A-Za-z0-9+&@#\/%=~_|]/i', '[REDACTED URL]', $message);

    return $message;
}

// تابع ایمن برای نوشتن در فایل‌های لاگ
function secureFilePutContents($filename, $data) {
    global $logDirectory;
    
    $allowedDir = realpath($logDirectory);
    $fileDir = realpath(dirname($filename));
    
    if ($fileDir !== $allowedDir || pathinfo($filename, PATHINFO_EXTENSION) !== 'log') {
        throw new Exception('Invalid log path');
    }

    $result = file_put_contents($filename, $data, FILE_APPEND | LOCK_EX);
    if ($result === false) {
        throw new Exception('Failed to write log');
    }
    return $result;
}

// تابع لاگ کردن خطاهای ربات (batch)
function logBotError($errorMessage, $userId = null, $chatId = null) {
    global $botErrorLogs;
    $timestamp = date('Y-m-d H:i:s');
    $safeErrorMessage = sanitizeLogData(sanitizeErrorMessage($errorMessage));
    if (empty($safeErrorMessage)) return;
    $context = '';
    if ($userId) $context .= " User: [REDACTED]";
    if ($chatId) $context .= " Chat: [REDACTED]";
    $botErrorLogs[] = "[{$timestamp}] {$safeErrorMessage}{$context}" . PHP_EOL;
}

// تابع لاگ کردن خطاهای عمومی (batch)
function logGeneralError($errorMessage) {
    global $generalErrorLogs;
    $timestamp = date('Y-m-d H:i:s');
    $safeErrorMessage = sanitizeLogData(sanitizeErrorMessage($errorMessage));
    if (empty($safeErrorMessage)) return;
    $generalErrorLogs[] = "[{$timestamp}] {$safeErrorMessage}" . PHP_EOL;
}

// تنظیمات Redis
$redisHost = '127.0.0.1';
$redisPort = 6379;
$redisTimeout = 1.0;

// تابع اتصال به Redis با مدیریت خطا
function getRedisConnection() {
    global $redisHost, $redisPort, $redisTimeout;
    try {
        $redis = new Redis();
        if (!$redis->pconnect($redisHost, $redisPort, $redisTimeout)) {
            throw new Exception("pconnect failed");
        }
        return $redis;
    } catch (Exception $e) {
        logGeneralError("Redis connection failed: " . $e->getMessage());
        return null;
    }
}

// ایجاد اتصال دیتابیس با تنظیمات امن و persistent connection pooling
try {
    $dsn = "mysql:host={$dbHost};dbname={$dbName};charset=utf8mb4";
    $options = array(
        PDO::ATTR_PERSISTENT => true,
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET sql_mode = 'STRICT_ALL_TABLES,NO_ENGINE_SUBSTITUTION'; SET time_zone = '+00:00'; SET wait_timeout = 300; SET interactive_timeout = 300;",
        PDO::MYSQL_ATTR_USE_BUFFERED_QUERY => true,
    );
    $db = new PDO($dsn, $dbUser, $dbPass, $options);
} catch (PDOException $e) {
    logGeneralError('Database connection failed: ' . $e->getMessage());
    exit('Database connection error');
}

// ایجاد جدول‌ها اگر وجود ندارند
try {
    $createUsersTable = "
    CREATE TABLE IF NOT EXISTS users (
        user_id BIGINT PRIMARY KEY,
        username VARCHAR(255),
        first_name VARCHAR(255),
        last_name VARCHAR(255),
        is_member BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $createUserRequestsTable = "
    CREATE TABLE IF NOT EXISTS user_requests (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        request_type VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX(user_id, created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $createUserMessagesTable = "
    CREATE TABLE IF NOT EXISTS user_messages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        message_text TEXT,
        is_malicious BOOLEAN DEFAULT FALSE,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX(user_id, created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $createBlockedUsersTable = "
    CREATE TABLE IF NOT EXISTS blocked_users (
        user_id BIGINT PRIMARY KEY,
        reason VARCHAR(255),
        blocked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $createStartCommandTable = "
    CREATE TABLE IF NOT EXISTS start_command_usage (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX(user_id, created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $createRequestQueueTable = "
    CREATE TABLE IF NOT EXISTS request_queue (
        id INT AUTO_INCREMENT PRIMARY KEY,
        method VARCHAR(50),
        params TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX(created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $db->exec($createUsersTable);
    $db->exec($createUserRequestsTable);
    $db->exec($createUserMessagesTable);
    $db->exec($createBlockedUsersTable);
    $db->exec($createStartCommandTable);
    $db->exec($createRequestQueueTable);
    
} catch (PDOException $e) {
    logGeneralError('Table creation error: ' . $e->getMessage());
}

// دریافت داده‌های ورودی با محدودیت سایز
$maxInputSize = 1024 * 1024;
$input = file_get_contents('php://input');
if (strlen($input) > $maxInputSize) {
    logGeneralError("Input size exceeded: " . strlen($input));
    exit;
}

$update = json_decode($input, true);

if (!$update) {
    logGeneralError("Invalid JSON input");
    exit;
}

// NEW: Queue incoming updates with Redis
$redis = getRedisConnection();
if ($redis) {
    $redis->rPush('incoming_queue', json_encode($update));
    
    processIncomingQueue($redis);
    
    $redis->close();
    exit;
} else {
    processUpdate($update);
}

// NEW: Function to process incoming queue with lock and retry
function processIncomingQueue($redis) {
    $lockKey = 'incoming_lock';
    $startTime = time();
    while (true) {
        $lockAcquired = $redis->set($lockKey, 'locked', array('NX', 'EX' => 30));
        
        if ($lockAcquired) {
            try {
                $queueLength = $redis->lLen('incoming_queue');
                if ($queueLength > 50) {
                    sendTelegramRequest('sendMessage', array(
                        'chat_id' => $GLOBALS['adminId'],
                        'text' => "⚠️ Queue length exceeded: {$queueLength}",
                        'parse_mode' => 'HTML'
                    ));
                }
                for ($i = 0; $i < 5; $i++) {  // کاهش به 5 آپدیت برای بهینه‌سازی
                    $rawUpdate = $redis->lPop('incoming_queue');
                    if (!$rawUpdate) break;
                    
                    $update = json_decode($rawUpdate, true);
                    if (!$update) continue;
                    
                    processUpdate($update);
                    
                    processRequestQueue(10, 1);  // کاهش max به 10
                }
            } finally {
                $redis->del($lockKey);  // همیشه release کن، حتی در exception
            }
            return;  // موفق، خارج شو
        } else {
            if (time() - $startTime > 10) {
                logGeneralError("Timeout acquiring lock, update remains in queue");
                return;  // صف می‌ماند برای بعدی
            }
            usleep(rand(100000, 500000));  // 100-500ms
        }
    }
}

// NEW: Extracted processing logic into a function
function processUpdate($update) {
    global $db, $adminId, $channel, $botToken, $supportUrl, $link1, $link2, $link3, $link4, $link5, $botErrorsLog, $errorLog, $logDirectory;

    if (isset($update['message'])) {
        $message = $update['message'];
        $chatId = $message['chat']['id'];
        $userId = $message['from']['id'];
        $text = $message['text'] ?? '';
        
        if (!is_numeric($userId) || $userId <= 0 || !is_numeric($chatId) || $chatId <= 0) {
            return;
        }
        
        $isAdminUnblockCommand = ($userId == $adminId && str_starts_with($text, '/unblock '));
        $isAdminClearCommand = ($userId == $adminId && $text == '/clear');
        
        if (!$isAdminUnblockCommand && !$isAdminClearCommand && !handleUserSafety($userId, $text, 'message')) {
            return;
        }
        
        try {
            $username = $message['from']['username'] ?? '';
            $firstName = $message['from']['first_name'] ?? '';
            $lastName = $message['from']['last_name'] ?? '';
            
            $safeUsername = htmlspecialchars(substr($username, 0, 255), ENT_QUOTES, 'UTF-8');
            $safeFirstName = htmlspecialchars(substr($firstName, 0, 255), ENT_QUOTES, 'UTF-8');
            $safeLastName = htmlspecialchars(substr($lastName, 0, 255), ENT_QUOTES, 'UTF-8');
            
            $stmt = $db->prepare("INSERT INTO users (user_id, username, first_name, last_name) VALUES (:user_id, :username, :first_name, :last_name) 
                                 ON DUPLICATE KEY UPDATE username = :username_upd, first_name = :first_name_upd, last_name = :last_name_upd");
            $stmt->execute(array(
                ':user_id' => $userId,
                ':username' => $safeUsername,
                ':first_name' => $safeFirstName,
                ':last_name' => $safeLastName,
                ':username_upd' => $safeUsername,
                ':first_name_upd' => $safeFirstName,
                ':last_name_upd' => $safeLastName
            ));
            
            // NEW: Invalidate cache for user info after update
            $redis = getRedisConnection();
            if ($redis) {
                $redis->del("user_info:{$userId}");
                $redis->close();
            }
        } catch (PDOException $e) {
            logGeneralError("User database operation failed: " . $e->getMessage());
        }
        
        if ($text == '/start') {
            sendLoadingMessage($chatId);
            
            if (userIsMemberInDB($userId)) {
                showMainMenu($chatId);
            } else {
                if (isChannelMember($userId, $channel)) {
                    try {
                        $stmt = $db->prepare("UPDATE users SET is_member = TRUE WHERE user_id = :user_id");
                        if ($stmt) {
                            $stmt->execute(array(':user_id' => $userId));
                        }
                    } catch (PDOException $e) {
                        logGeneralError("Failed to update user membership status: " . $e->getMessage());
                    }
                    
                    showMainMenu($chatId);
                } else {
                    showChannelMenu($chatId);
                }
            }
        }
        elseif ($text && $userId == $adminId && str_starts_with($text, '/unblock ')) {
            $targetUserId = substr($text, 9);
            
            if (is_numeric($targetUserId) && $targetUserId > 0) {
                $success = unblockUser($targetUserId);
                
                if ($success) {
                    sendTelegramRequest('sendMessage', array(
                        'chat_id' => $chatId,
                        'text' => "✅ کاربر با شناسه {$targetUserId} آزاد شد.",
                        'parse_mode' => 'HTML'
                    ));
                    
                    sendUnblockNotificationToAdmin($targetUserId, true);
                } else {
                    sendTelegramRequest('sendMessage', array(
                        'chat_id' => $chatId,
                        'text' => "❌ کاربر با شناسه {$targetUserId} پیدا نشد یا از قبل آزاد بود.",
                        'parse_mode' => 'HTML'
                    ));
                }
            } else {
                sendTelegramRequest('sendMessage', array(
                    'chat_id' => $chatId,
                    'text' => "❌ شناسه کاربر باید عددی مثبت باشد.\n\nاستفاده: /unblock 123456789",
                    'parse_mode' => 'HTML'
                ));
            }
        }
        elseif ($text == '/clear' && $userId == $adminId) {
            if (file_exists($botErrorsLog)) {
                file_put_contents($botErrorsLog, '');
            }
            if (file_exists($errorLog)) {
                file_put_contents($errorLog, '');
            }

            $tables = array(
                'users',
                'user_requests',
                'user_messages',
                'blocked_users',
                'start_command_usage'
            );

            foreach ($tables as $table) {
                $db->exec("TRUNCATE TABLE $table");
            }

            // NEW: Clear all Redis caches on /clear
            $redis = getRedisConnection();
            if ($redis) {
                $redis->flushDB();  // Clear all keys (be careful in production)
                $redis->close();
            }

            sendTelegramRequest('sendMessage', array(
                'chat_id' => $chatId,
                'text' => "✅ تمام محتوای فایل‌های لاگ و دیتابیس پاک شد.",
                'parse_mode' => 'HTML'
            ));
        }
    }

    if (isset($update['callback_query'])) {
        $callback = $update['callback_query'];
        $userId = $callback['from']['id'];
        $chatId = $callback['message']['chat']['id'];
        $messageId = $callback['message']['message_id'];
        $data = $callback['data'];
        
        if (!is_numeric($userId) || $userId <= 0 || !is_numeric($chatId) || $chatId <= 0) {
            return;
        }
        
        $isAdminUnblockCallback = (str_starts_with($data, 'unblock_user:') && $userId == $adminId);
        
        if (!$isAdminUnblockCallback && !handleUserSafety($userId, '', 'callback')) {
            return;
        }
        
        if ($data == 'check_membership') {
            if (isChannelMember($userId, $channel)) {
                try {
                    $stmt = $db->prepare("UPDATE users SET is_member = TRUE WHERE user_id = :user_id");
                    if ($stmt) {
                        $stmt->execute(array(':user_id' => $userId));
                    }
                } catch (PDOException $e) {
                    logGeneralError("Failed to update user membership status in callback: " . $e->getMessage());
                }
                
                sendTelegramRequest('deleteMessage', array(
                    'chat_id' => $chatId,
                    'message_id' => $messageId
                ));
                showMainMenu($chatId);
                
                sendTelegramRequest('answerCallbackQuery', array(
                    'callback_query_id' => $callback['id'],
                    'text' => '✅  عضویت شما تایید شد!'
                ));
            } else {
                $alertText = "❌  شما هنوز عضو کانال نشده اید.\n\n❗ برای مشاهده اطلاعیه ها، تخفیف ها و آموزش ها، بهتر است ابتدا در کانال پشتیبانی عضو شوید.";
                
                sendTelegramRequest('answerCallbackQuery', array(
                    'callback_query_id' => $callback['id'],
                    'text' => $alertText,
                    'show_alert' => true
                ));
            }
        }
        elseif ($data == 'back_to_main') {
            sendTelegramRequest('deleteMessage', array(
                'chat_id' => $chatId,
                'message_id' => $messageId
            ));
            showMainMenu($chatId);
            
            sendTelegramRequest('answerCallbackQuery', array(
                'callback_query_id' => $callback['id']
            ));
        }
        elseif ($data == 'test_account') {
            $alertText = "🔬 اکانت تست 3 گیگ 5 روزه = 10 هزار تومان\n\n💳  برای خرید به پشتیبانی پیام دهید.";
            
            sendTelegramRequest('answerCallbackQuery', array(
                'callback_query_id' => $callback['id'],
                'text' => $alertText,
                'show_alert' => true
            ));
        }
        elseif ($data == 'tutorial') {
            showTutorialPanel($chatId, $messageId);
            sendTelegramRequest('answerCallbackQuery', array(
                'callback_query_id' => $callback['id']
            ));
        }
        elseif ($data == 'telegram_premium') {
            showTelegramPremiumPanel($chatId, $messageId);
            sendTelegramRequest('answerCallbackQuery', array(
                'callback_query_id' => $callback['id']
            ));
        }
        elseif (str_starts_with($data, 'unblock_user:')) {
            if ($userId == $adminId) {
                $targetUserId = substr($data, 13);
                
                if (is_numeric($targetUserId) && $targetUserId > 0) {
                    $success = unblockUser($targetUserId);
                    
                    sendUnblockNotificationToAdmin($targetUserId, $success);
                    
                    $originalMessage = $callback['message']['text'];
                    $updatedMessage = $originalMessage . "\n\n✅ <b>آزاد شده توسط ادمین</b>";
                    
                    sendTelegramRequest('editMessageText', array(
                        'chat_id' => $chatId,
                        'message_id' => $messageId,
                        'text' => $updatedMessage,
                        'parse_mode' => 'HTML'
                    ));
                    
                    sendTelegramRequest('answerCallbackQuery', array(
                        'callback_query_id' => $callback['id'],
                        'text' => $success ? '✅ کاربر آزاد شد' : '❌ خطا در آزاد کردن کاربر'
                    ));
                } else {
                    sendTelegramRequest('answerCallbackQuery', array(
                        'callback_query_id' => $callback['id'],
                        'text' => '❌ شناسه کاربر نامعتبر است',
                        'show_alert' => true
                    ));
                }
            } else {
                sendTelegramRequest('answerCallbackQuery', array(
                    'callback_query_id' => $callback['id'],
                    'text' => '❌ فقط ادمین می‌تواند این کار را انجام دهد',
                    'show_alert' => true
                ));
            }
        }
    }
    
    processRequestQueue(10, 1);  // کاهش max به 10
}

// تابع Rate Limiting با Redis (اصلی) + Fallback به MySQL Transaction
function isRateLimited($userId, $maxRequests = 25, $timeWindow = 60) {
    global $db, $adminId;
    
    if (!is_numeric($userId) || $userId <= 0) {
        return true;
    }
    
    if ($userId == $adminId) {
        return false;
    }
    
    $redis = getRedisConnection();
    if ($redis) {
        try {
            $key = "rate_limit:user:{$userId}";
            $current = $redis->incr($key);
            if ($current == 1) {
                $redis->expire($key, $timeWindow);
            }
            $redis->close();
            return $current > $maxRequests;
        } catch (Exception $e) {
            logGeneralError("Redis rate limit error: " . $e->getMessage());
        }
    }
    
    // Fallback to MySQL (no additional cache here as it's already time-based)
    $db->beginTransaction();
    try {
        $stmt = $db->prepare("INSERT INTO user_requests (user_id, request_type) VALUES (:user_id, 'message')");
        $stmt->execute(array(':user_id' => $userId));
        
        $timeAgo = date('Y-m-d H:i:s', time() - $timeWindow);
        $stmt = $db->prepare("SELECT COUNT(*) as request_count FROM user_requests 
                             WHERE user_id = :user_id AND created_at > :time_ago FOR UPDATE");
        $stmt->execute(array(':user_id' => $userId, ':time_ago' => $timeAgo));
        $row = $stmt->fetch();
        $count = $row['request_count'] ?? 0;
        $db->commit();
        
        return $count >= $maxRequests;
    } catch (PDOException $e) {
        $db->rollBack();
        logGeneralError("MySQL rate limit transaction failed: " . $e->getMessage());
        return true;
    }
}

// تابع مشابه برای Start Command Rate Limiting
function isStartCommandRateLimited($userId, $maxStarts = 5, $timeWindow = 3600) {
    global $db, $adminId;
    
    if (!is_numeric($userId) || $userId <= 0 || $userId == $adminId) {
        return false;
    }
    
    $redis = getRedisConnection();
    if ($redis) {
        try {
            $key = "start_limit:user:{$userId}";
            $current = $redis->incr($key);
            if ($current == 1) {
                $redis->expire($key, $timeWindow);
            }
            $redis->close();
            return $current > $maxStarts;
        } catch (Exception $e) {
            logGeneralError("Redis start limit error: " . $e->getMessage());
        }
    }
    
    $db->beginTransaction();
    try {
        $stmt = $db->prepare("INSERT INTO start_command_usage (user_id) VALUES (:user_id)");
        $stmt->execute(array(':user_id' => $userId));
        
        $timeAgo = date('Y-m-d H:i:s', time() - $timeWindow);
        $stmt = $db->prepare("SELECT COUNT(*) as start_count FROM start_command_usage 
                             WHERE user_id = :user_id AND created_at > :time_ago FOR UPDATE");
        $stmt->execute(array(':user_id' => $userId, ':time_ago' => $timeAgo));
        $row = $stmt->fetch();
        $count = $row['start_count'] ?? 0;
        $db->commit();
        
        return $count >= $maxStarts;
    } catch (PDOException $e) {
        $db->rollBack();
        logGeneralError("MySQL start limit transaction failed: " . $e->getMessage());
        return true;
    }
}

// تابع جدید: Per-chat rate limiter با Redis (حذف usleepهای غیرضروری)
function perChatRateLimiter($chatId) {
    $maxChatRequests = 1;
    $chatWindowSeconds = 1;

    $redis = getRedisConnection();
    if ($redis) {
        $key = 'chat_rate_limit:' . $chatId;
        $currentTime = microtime(true);
        
        $redis->zRemRangeByScore($key, '-inf', $currentTime - $chatWindowSeconds);
        $count = $redis->zCard($key);
        
        while ($count >= $maxChatRequests) {
            $range = $redis->zRange($key, 0, 0, true);
            $oldestScore = $range ? reset($range) : $currentTime;
            $sleepTime = ($oldestScore + $chatWindowSeconds - $currentTime) + 0.1;
            if ($sleepTime > 0) {
                usleep((int)($sleepTime * 1000000));
            }
            $redis->zRemRangeByScore($key, '-inf', microtime(true) - $chatWindowSeconds);
            $count = $redis->zCard($key);
        }
        
        $redis->zAdd($key, $currentTime, uniqid());
        $redis->expire($key, $chatWindowSeconds * 2);
        $redis->close();
    } else {
        // حذف usleep برای بهینه‌سازی
    }
}

// IMPROVED: Global Rate Limiter بدون Redis
function globalRateLimit($max = 20, $window = 1) {
    $file = sys_get_temp_dir() . '/bot_global_rate.lock';
    $fp = fopen($file, 'c+');
    if ($fp === false) {
        logGeneralError("Failed to open global rate limit file");
        return;
    }
    
    if (flock($fp, LOCK_EX)) {
        rewind($fp);
        $data = stream_get_contents($fp);
        $timestamps = $data ? json_decode($data, true) : array();
        $now = microtime(true);
        $timestamps = array_filter($timestamps, function($t) use ($now, $window) {
            return $t > $now - $window;
        });
        
        if (count($timestamps) >= $max) {
            $oldest = min($timestamps);
            $wait = ($oldest + $window) - $now;
            flock($fp, LOCK_UN);
            fclose($fp);
            usleep((int)(max(0, $wait) * 1000000 + 50000));
            return globalRateLimit($max, $window);
        }
        
        $timestamps[] = $now;
        ftruncate($fp, 0);
        rewind($fp);
        fwrite($fp, json_encode(array_slice($timestamps, -50)));
        fflush($fp);
        flock($fp, LOCK_UN);
    } else {
        logGeneralError("Failed to lock global rate limit file");
    }
    
    fclose($fp);
}

// IMPROVED: Per-chat Delay برای همه درخواست‌ها
function perChatDelay($chatId) {
    $file = sys_get_temp_dir() . "/chat_{$chatId}.lock";
    $fp = fopen($file, 'c+');
    if ($fp === false) {
        logGeneralError("Failed to open per-chat delay file for chat {$chatId}");
        return;
    }
    
    if (flock($fp, LOCK_EX)) {
        rewind($fp);
        $last = stream_get_contents($fp);
        $now = microtime(true);
        if ($last && ($now - (float)$last) < 1.0) {
            $sleep = ((float)$last + 1.0 - $now);
            flock($fp, LOCK_UN);
            fclose($fp);
            usleep((int)($sleep * 1000000));
            return perChatDelay($chatId);
        }
        
        ftruncate($fp, 0);
        rewind($fp);
        fwrite($fp, (string)$now);
        fflush($fp);
        flock($fp, LOCK_UN);
    } else {
        logGeneralError("Failed to lock per-chat delay file for chat {$chatId}");
    }
    
    fclose($fp);
}

// تابع جدید: چک اینکه آیا اطلاع‌رسانی ریت لیمیت اخیراً ارسال شده
function shouldNotifyAdminRateLimit() {
    static $lastNotifyTime = 0;
    $now = time();
    if ($now - $lastNotifyTime < 60) {
        return false;
    }
    
    $redis = getRedisConnection();
    if ($redis) {
        $key = 'rate_limit_notify_cooldown';
        if ($redis->exists($key)) {
            $redis->close();
            return false;
        }
        $redis->set($key, '1', array('EX' => 60));
        $redis->close();
        return true;
    }
    
    $file = sys_get_temp_dir() . '/rate_limit_notify.lock';
    if (file_exists($file) && (time() - filemtime($file)) < 60) {
        return false;
    }
    touch($file);
    $lastNotifyTime = $now;
    return true;
}

// تابع ارسال درخواست به تلگرام با SSL همیشه فعال و مدیریت rate limit
function sendTelegramRequest($method, $params = array(), $maxRetries = 2) {  // کاهش maxRetries به 2
    global $botToken, $logDirectory, $db, $adminId;

    $chatId = $params['chat_id'] ?? null;
    if ($chatId) {
        perChatDelay($chatId);
        perChatRateLimiter($chatId);
    }

    globalRateLimit(20, 1);

    $url = "https://api.telegram.org/bot{$botToken}/{$method}";
    
    $windowSeconds = 1;
    $defaultMax = 20;
    $minMax = 10;

    $redis = getRedisConnection();
    if ($redis) {
        $maxKey = 'global_rate_max';
        $currentMax = (int)$redis->get($maxKey);
        if (!$currentMax) {
            $currentMax = $defaultMax;
            $redis->set($maxKey, $currentMax);
        }
        
        $key = 'global_rate_limit';
        $currentTime = microtime(true);
        
        $script = "
        local key = KEYS[1]
        local now = tonumber(ARGV[1])
        local window = tonumber(ARGV[2])
        local limit = tonumber(ARGV[3])
        redis.call('ZREMRANGEBYSCORE', key, '-inf', now - window)
        local count = redis.call('ZCARD', key)
        if count < limit then
            redis.call('ZADD', key, now, now .. math.random())
            redis.call('EXPIRE', key, window * 2)
            return {1, 0}
        else
            local oldest = redis.call('ZRANGE', key, 0, 0, 'WITHSCORES')
            local oldest_score = 0
            if oldest[2] then
                oldest_score = oldest[2]
            end
            return {0, oldest_score}
        end
        ";
        $res = $redis->eval($script, array($key, $currentTime, $windowSeconds, $currentMax), 1);
        
        if (is_array($res) && $res[0] == 0) {
            $oldestScore = (float)$res[1];
            $sleepTime = ($oldestScore + $windowSeconds - $currentTime) + 0.05;
            if ($sleepTime > 0) {
                usleep((int)($sleepTime * 1000000));
            }
            $currentTime = microtime(true);
            $res = $redis->eval($script, array($key, $currentTime, $windowSeconds, $currentMax), 1);
        }
        
        $redis->close();
    } else {
        $serializedParams = json_encode($params);
        $stmt = $db->prepare("INSERT INTO request_queue (method, params) VALUES (:method, :params)");
        $stmt->execute(array(':method' => $method, ':params' => $serializedParams));

        processRequestQueue($defaultMax, $windowSeconds);

        return;
    }

    $retryCount = 0;
    $backoff = 1;
    
    while ($retryCount < $maxRetries) {
        $ch = curl_init();
        
        $curlOptions = array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POSTFIELDS => $params,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_FOLLOWLOCATION => false,
            CURLOPT_HTTPHEADER => array(
                'User-Agent: Telegram Bot PHP Client'
            )
        );
        
        $possibleCertPaths = array(
            '/etc/ssl/certs/ca-certificates.crt',
            '/etc/pki/tls/certs/ca-bundle.crt',
            '/usr/share/ssl/certs/ca-bundle.crt',
            '/etc/ssl/certs/ca-bundle.crt',
            '/etc/pki/tls/cacert.pem',
            '/etc/ssl/cert.pem'
        );
        
        foreach ($possibleCertPaths as $certPath) {
            if (file_exists($certPath)) {
                $curlOptions[CURLOPT_CAINFO] = $certPath;
                break;
            }
        }
        
        if (!isset($curlOptions[CURLOPT_CAINFO])) {
            $curlOptions[CURLOPT_CAPATH] = '/etc/ssl/certs';
        }
        
        curl_setopt_array($ch, $curlOptions);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);
        $curlErrno = curl_errno($ch);
        curl_close($ch);
        
        if ($response === false) {
            logBotError("cURL Error ({$curlErrno}): {$curlError}", null, $chatId);
            return false;
        }
        
        if ($httpCode === 200) {
            usleep(50000);
            return $response;
        } elseif ($httpCode === 429) {
            $data = json_decode($response, true);
            $retryAfter = isset($data['parameters']['retry_after']) ? $data['parameters']['retry_after'] : $backoff;
            logBotError("Rate limit hit: HTTP 429 - Retry after {$retryAfter} seconds", null, $chatId);
            
            if (shouldNotifyAdminRateLimit()) {
                $notifyParams = array(
                    'chat_id' => $adminId,
                    'text' => "⚠️ Global Rate Limit violated: HTTP 429 - Retry after {$retryAfter} seconds",
                    'parse_mode' => 'HTML'
                );
                $serializedNotifyParams = json_encode($notifyParams);
                $stmt = $db->prepare("INSERT INTO request_queue (method, params) VALUES ('sendMessage', :params)");
                $stmt->execute(array(':params' => $serializedNotifyParams));
            }
            
            $redis = getRedisConnection();
            if ($redis) {
                $maxKey = 'global_rate_max';
                $currentMax = (int)$redis->get($maxKey) ?: $defaultMax;
                $newMax = max($minMax, $currentMax - 1);
                $redis->set($maxKey, $newMax, array('EX' => 60));
                $redis->close();
            }
            
            if ($retryAfter > 60) {
                logBotError("Floodwait too long ({$retryAfter}s), skipping request", null, $chatId);
                return false;
            }
            sleep((int)($retryAfter + rand(0, 1)));
            $backoff *= 2;
            $retryCount++;
        } else {
            logBotError("HTTP Error: {$httpCode} - Response: " . substr($response, 0, 500), null, $chatId);
            return false;
        }
    }
    
    logBotError("Max retries exceeded for method: {$method}", null, $chatId);
    return false;
}

// NEW: تابع پردازش queue (کاهش limit به 10 و استفاده از single curl برای سادگی)
function processRequestQueue($maxRequests, $windowSeconds) {
    global $db, $botToken;

    $stmt = $db->prepare("SELECT id, method, params FROM request_queue ORDER BY created_at ASC LIMIT :max_requests");
    $stmt->bindValue(':max_requests', $maxRequests, PDO::PARAM_INT);
    $stmt->execute();
    $requests = $stmt->fetchAll();

    if (empty($requests)) return;

    foreach ($requests as $req) {  // استفاده از single curl به جای multi برای کاهش فرآیندها
        $params = json_decode($req['params'], true);
        $url = "https://api.telegram.org/bot{$botToken}/{$req['method']}";

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('User-Agent: Telegram Bot PHP Client'));
        
        $possibleCertPaths = array(
            '/etc/ssl/certs/ca-certificates.crt',
            '/etc/pki/tls/certs/ca-bundle.crt',
            '/usr/share/ssl/certs/ca-bundle.crt',
            '/etc/ssl/certs/ca-bundle.crt',
            '/etc/pki/tls/cacert.pem',
            '/etc/ssl/cert.pem'
        );
        foreach ($possibleCertPaths as $certPath) {
            if (file_exists($certPath)) {
                curl_setopt($ch, CURLOPT_CAINFO, $certPath);
                break;
            }
        }
        if (!curl_getinfo($ch, CURLOPT_CAINFO)) {
            curl_setopt($ch, CURLOPT_CAPATH, '/etc/ssl/certs');
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($httpCode === 200) {
        } else {
            logBotError("Queue request failed: HTTP {$httpCode} for method {$req['method']}");
        }

        curl_close($ch);

        $stmt = $db->prepare("DELETE FROM request_queue WHERE id = :id");
        $stmt->execute(array(':id' => $req['id']));
    }

    usleep((int)(1000000 / $maxRequests));
}

// تابع تبدیل متن به Bold با استفاده از HTML با escaping امن
function makeTextBold($text) {
    return "<b>" . htmlspecialchars($text, ENT_QUOTES, 'UTF-8') . "</b>";
}

// تابع ارسال پیام ⏳ و نگهداری آن
function sendLoadingMessage($chatId) {
    $messageText = "⏳";
    
    $result = sendTelegramRequest('sendMessage', array(
        'chat_id' => $chatId,
        'text' => $messageText,
        'parse_mode' => 'HTML'
    ));
    
    if ($result === false) {
        logBotError("Failed to send loading message to chat ID: {$chatId}");
        return null;
    }
    
    $response = json_decode($result, true);
    if ($response && $response['ok']) {
        return $response['result']['message_id'];
    }
    
    return null;
}

// تابع برای چک str_contains_array
function str_contains_array($haystack, $needles) {
    foreach ($needles as $needle) {
        if (stripos($haystack, $needle) !== false) {
            return true;
        }
    }
    return false;
}

// تابع تشخیص الگوهای مخرب - بهبود یافته
function isMaliciousPattern($text) {
    if (empty($text) || strlen($text) > 5000) return true;
    
    if (preg_match('/[^\x{0600}-\x{06FF}\x{0000}-\x{007F}\s\.\,\-\_\@\:\/\(\)\!\?\+\=\*\%\$\#\~\`\&\{\}\[\]\<\>]/u', $text)) {
        return true;
    }
    
    $quickChecks = array(
        '<script', 'javascript:', 'vbscript:', 'data:', 'blob:',
        'union select', 'insert into', 'update set', 'delete from', 'drop table',
        'exec(', 'execute(', 'sys.', 'os.', ';', '&&', '../', '..\\',
        '<?php', 'document.cookie', 'eval(', 'include(', 'system(', '%0a', '<!--#'
    );
    if (str_contains_array($text, $quickChecks)) {
        return true;
    }
    
    return false;
}

// تابع بررسی کاربر بلاک شده (با caching)
function isUserBlocked($userId) {
    global $db;
    
    if (!is_numeric($userId) || $userId <= 0) {
        return true;
    }
    
    $redis = getRedisConnection();
    if ($redis) {
        $cacheKey = "blocked:user:{$userId}";
        $cached = $redis->get($cacheKey);
        if ($cached !== false) {
            $redis->close();
            return (bool)json_decode($cached);
        }
    }
    
    $stmt = $db->prepare("SELECT user_id FROM blocked_users WHERE user_id = :user_id");
    if (!$stmt) {
        logGeneralError("Failed to prepare blocked user check statement: " . $db->errorInfo()[2]);
        return false;
    }
    
    $stmt->execute(array(':user_id' => $userId));
    $result = $stmt->fetchAll();
    $isBlocked = count($result) > 0;
    
    if ($redis) {
        $redis->set($cacheKey, json_encode($isBlocked), array('EX' => 60));  // Cache for 60 seconds
        $redis->close();
    }
    
    return $isBlocked;
}

// تابع بلاک کردن کاربر (با invalidation cache)
function blockUser($userId, $reason) {
    global $db, $adminId;
    
    if (!is_numeric($userId) || $userId <= 0) {
        return false;
    }
    
    if ($userId == $adminId) {
        return false;
    }
    
    $safeReason = htmlspecialchars(substr($reason, 0, 250), ENT_QUOTES, 'UTF-8');
    
    $stmt = $db->prepare("INSERT INTO blocked_users (user_id, reason) VALUES (:user_id, :reason)");
    if (!$stmt) {
        logGeneralError("Failed to prepare block user statement: " . $db->errorInfo()[2]);
        return false;
    }
    
    $success = $stmt->execute(array(':user_id' => $userId, ':reason' => $safeReason));
    
    if ($success) {
        logGeneralError("User blocked: {$userId}, Reason: {$safeReason}");
        sendBlockNotificationToAdmin($userId, $safeReason);
        
        // NEW: Invalidate cache
        $redis = getRedisConnection();
        if ($redis) {
            $redis->del("blocked:user:{$userId}");
            $redis->close();
        }
        
        return true;
    }
    
    return false;
}

// تابع ارسال اطلاعیه بلاک به ادمین
function sendBlockNotificationToAdmin($blockedUserId, $reason) {
    global $adminId, $db;
    
    if (!is_numeric($blockedUserId) || $blockedUserId <= 0 || empty($adminId)) {
        return;
    }
    
    // NEW: Cache user info for notifications
    $redis = getRedisConnection();
    $cacheKey = "user_info:{$blockedUserId}";
    if ($redis) {
        $cachedUser = $redis->get($cacheKey);
        if ($cachedUser !== false) {
            $user = json_decode($cachedUser, true);
            $redis->close();
        }
    }
    
    if (!isset($user)) {
        $stmt = $db->prepare("SELECT username, first_name, last_name FROM users WHERE user_id = :user_id");
        if (!$stmt) {
            return;
        }
        
        $stmt->execute(array(':user_id' => $blockedUserId));
        $user = $stmt->fetch();
        
        if ($redis) {
            $redis->set($cacheKey, json_encode($user), array('EX' => 300));  // Cache for 5 minutes
            $redis->close();
        }
    }
    
    $username = $user['username'] ?? 'ندارد';
    $firstName = htmlspecialchars($user['first_name'] ?? 'ندارد', ENT_QUOTES, 'UTF-8');
    $lastName = htmlspecialchars($user['last_name'] ?? 'ندارد', ENT_QUOTES, 'UTF-8');
    
    $messageText = "🚨 <b>کاربر بلاک شد</b>\n\n";
    $messageText .= "🆔 <b>User ID:</b> <code>{$blockedUserId}</code>\n";
    $messageText .= "👤 <b>Username:</b> @" . htmlspecialchars($username, ENT_QUOTES, 'UTF-8') . "\n";
    $messageText .= "📛 <b>First Name:</b> {$firstName}\n";
    $messageText .= "📛 <b>Last Name:</b> {$lastName}\n";
    $messageText .= "⚠️ <b>Reason:</b> {$reason}\n";
    $messageText .= "🕒 <b>Time:</b> " . date('Y-m-d H:i:s');
    
    $keyboard = array(
        'inline_keyboard' => array(
            array(
                array('text' => '🔓 آزاد کردن کاربر', 'callback_data' => 'unblock_user:' . $blockedUserId)
            )
        )
    );
    
    $result = sendTelegramRequest('sendMessage', array(
        'chat_id' => $adminId,
        'text' => $messageText,
        'reply_markup' => json_encode($keyboard),
        'parse_mode' => 'HTML'
    ));
    
    if ($result === false) {
        logBotError("Failed to send block notification to admin");
    }
}

// تابع آزاد کردن کاربر (با invalidation cache)
function unblockUser($userId) {
    global $db;
    
    if (!is_numeric($userId) || $userId <= 0) {
        return false;
    }
    
    $stmt = $db->prepare("DELETE FROM blocked_users WHERE user_id = :user_id");
    if (!$stmt) {
        logGeneralError("Failed to prepare unblock user statement: " . $db->errorInfo()[2]);
        return false;
    }
    
    $success = $stmt->execute(array(':user_id' => $userId));
    
    if ($success) {
        $stmt = $db->prepare("DELETE FROM user_requests WHERE user_id = :user_id");
        if ($stmt) {
            $stmt->execute(array(':user_id' => $userId));
        }
        
        $stmt = $db->prepare("DELETE FROM start_command_usage WHERE user_id = :user_id");
        if ($stmt) {
            $stmt->execute(array(':user_id' => $userId));
        }
        
        logGeneralError("User unblocked: {$userId}");
        
        // NEW: Invalidate cache
        $redis = getRedisConnection();
        if ($redis) {
            $redis->del("blocked:user:{$userId}");
            $redis->close();
        }
        
        return true;
    }
    
    return false;
}

// تابع ارسال اطلاعیه آزادسازی به ادمین
function sendUnblockNotificationToAdmin($unblockedUserId, $success) {
    global $adminId, $db;
    
    if (!is_numeric($unblockedUserId) || $unblockedUserId <= 0 || empty($adminId)) {
        return;
    }
    
    // NEW: Use cached user info
    $redis = getRedisConnection();
    $cacheKey = "user_info:{$unblockedUserId}";
    if ($redis) {
        $cachedUser = $redis->get($cacheKey);
        if ($cachedUser !== false) {
            $user = json_decode($cachedUser, true);
            $redis->close();
        }
    }
    
    if (!isset($user)) {
        $stmt = $db->prepare("SELECT username, first_name, last_name FROM users WHERE user_id = :user_id");
        if (!$stmt) {
            return;
        }
        
        $stmt->execute(array(':user_id' => $unblockedUserId));
        $user = $stmt->fetch();
        
        if ($redis) {
            $redis->set($cacheKey, json_encode($user), array('EX' => 300));
            $redis->close();
        }
    }
    
    $username = $user['username'] ?? 'ندارد';
    $firstName = htmlspecialchars($user['first_name'] ?? 'ندارد', ENT_QUOTES, 'UTF-8');
    $lastName = htmlspecialchars($user['last_name'] ?? 'ندارد', ENT_QUOTES, 'UTF-8');
    
    if ($success) {
        $messageText = "✅ <b>کاربر آزاد شد</b>\n\n";
        $messageText .= "🆔 <b>User ID:</b> <code>{$unblockedUserId}</code>\n";
        $messageText .= "👤 <b>Username:</b> @" . htmlspecialchars($username, ENT_QUOTES, 'UTF-8') . "\n";
        $messageText .= "📛 <b>First Name:</b> {$firstName}\n";
        $messageText .= "📛 <b>Last Name:</b> {$lastName}\n";
        $messageText .= "🕒 <b>Time:</b> " . date('Y-m-d H:i:s');
    } else {
        $messageText = "❌ <b>خطا در آزاد کردن کاربر</b>\n\n";
        $messageText .= "🆔 <b>User ID:</b> <code>{$unblockedUserId}</code>\n";
        $messageText .= "👤 <b>Username:</b> @" . htmlspecialchars($username, ENT_QUOTES, 'UTF-8') . "\n";
    }
    
    $result = sendTelegramRequest('sendMessage', array(
        'chat_id' => $adminId,
        'text' => $messageText,
        'parse_mode' => 'HTML'
    ));
    
    if ($result === false) {
        logBotError("Failed to send unblock notification to admin");
    }
}

// NEW: تابع چک عضویت کاربر در DB (بدون cache طبق درخواست)
function userIsMemberInDB($userId) {
    global $db;
    $stmt = $db->prepare("SELECT is_member FROM users WHERE user_id = :user_id");
    $stmt->execute(array(':user_id' => $userId));
    $row = $stmt->fetch();
    return $row['is_member'] ?? false;
}

// تابع بهبود یافته مدیریت امنیت کاربر - برای تمام انواع درخواست‌ها (batch insert برای messages و requests)
function handleUserSafety($userId, $messageText = '', $requestType = 'message') {
    global $adminId, $db;
    
    if ($userId == $adminId) {
        if (!empty($messageText) && $requestType === 'message') {
            $safeMessageText = htmlspecialchars(substr($messageText, 0, 1000), ENT_QUOTES, 'UTF-8');
            
            $stmt = $db->prepare("INSERT INTO user_messages (user_id, message_text) VALUES (:user_id, :message_text)");
            if ($stmt) {
                $stmt->execute(array(':user_id' => $userId, ':message_text' => $safeMessageText));
            }
        }
        
        return true;
    }
    
    if (isUserBlocked($userId)) {
        return false;
    }
    
    if (isRateLimited($userId)) {
        blockUser($userId, "Rate limiting violation");
        return false;
    }
    
    if (!empty($messageText) && $messageText == '/start' && $requestType === 'message') {
        if (isStartCommandRateLimited($userId)) {
            blockUser($userId, "Start command rate limiting violation");
            return false;
        }
    }
    
    $isMalicious = !empty($messageText) && $requestType === 'message' && isMaliciousPattern($messageText);
    
    if ($isMalicious) {
        blockUser($userId, "Malicious pattern detected");
    }
    
    if (!empty($messageText) && $requestType === 'message') {
        $safeMessageText = htmlspecialchars(substr($messageText, 0, 1000), ENT_QUOTES, 'UTF-8');
        
        $stmt = $db->prepare("INSERT INTO user_messages (user_id, message_text, is_malicious) VALUES (:user_id, :message_text, :is_malicious)");
        if ($stmt) {
            $stmt->execute(array(':user_id' => $userId, ':message_text' => $safeMessageText, ':is_malicious' => (int)$isMalicious));
        }
    }
    
    return !$isMalicious;
}

// تابع نمایش منوی اصلی
function showMainMenu($chatId) {
    global $supportUrl, $link1, $link2;
    
    $safeSupportUrl = filter_var($supportUrl, FILTER_VALIDATE_URL) ? $supportUrl : '';
    $safeLink1 = filter_var($link1, FILTER_VALIDATE_URL) ? $link1 : '';
    $safeLink2 = filter_var($link2, FILTER_VALIDATE_URL) ? $link2 : '';
    
    $keyboard = array(
        'inline_keyboard' => array(
            array(
                array('text' => '1⃣  سرویس V2ray تک لوکیشن', 'url' => $safeLink1)
            ),
            array(
                array('text' => '2⃣  سرویس V2ray مولتی لوکیشن', 'url' => $safeLink2)
            ),
            array(
                array('text' => '👤  پشتیبانی', 'url' => $safeSupportUrl),
                array('text' => '🥚  اکانت تست', 'callback_data' => 'test_account')
            ),
            array(
                array('text' => '🧩   لیست نرم افزارها', 'callback_data' => 'tutorial')
            ),
            array(
                array('text' => '⭐  تلگرام پرمیوم  ⭐', 'callback_data' => 'telegram_premium')
            )
        )
    );
    
    $messageText = "🟢  سرویسهای V2ray با لوکیشن های آلمان 🇩🇪، هلند 🇳🇱، آمریکا 🇺🇸، انگلیس 🇬🇧، ترکیه 🇹🇷، فنلاند 🇫🇮، فرانسه 🇫🇷\n\n✅  آیپی ثابت، سرعت موشکی و پینگ عالی\n✅  فعال با تمام اپراتورها\n✅  آندروید، آیفون، کامپیوتر\n✅  پشتیبانی فنی تا آخرین روز اشتراک\n✅  عبور از محدودیت Chat Gpt ، Gemini، Grok ، CapCut و ...";
    $messageText = makeTextBold($messageText);
    
    $result = sendTelegramRequest('sendMessage', array(
        'chat_id' => $chatId,
        'text' => $messageText,
        'reply_markup' => json_encode($keyboard),
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ));
    
    if ($result === false) {
        logBotError("Failed to show main menu to chat ID: {$chatId}");
    }
}

// تابع نمایش منوی کانال
function showChannelMenu($chatId) {
    global $channelUrl;
    
    $safeChannelUrl = filter_var($channelUrl, FILTER_VALIDATE_URL) ? $channelUrl : '';
    
    $keyboard = array(
        'inline_keyboard' => array(
            array(
                array('text' => '🌍  عضویت در کانال', 'url' => $safeChannelUrl)
            ),
            array(
                array('text' => '✅  عضو شدم', 'callback_data' => 'check_membership')
            )
        )
    );
    
    $messageText = "❗️ کاربر گرامی ؛\n\nبرای مشاهده اطلاعیه ها، تخفیف ها و آموزش ها ، بهتر است ابتدا در کانال پشتیبانی عضو شوید.";
    $messageText = makeTextBold($messageText);
    
    $result = sendTelegramRequest('sendMessage', array(
        'chat_id' => $chatId,
        'text' => $messageText,
        'reply_markup' => json_encode($keyboard),
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => true
    ));
    
    if ($result === false) {
        logBotError("Failed to show channel menu to chat ID: {$chatId}");
    }
}

// تابع بررسی عضویت در کانال (بدون cache طبق درخواست)
function isChannelMember($userId, $channel) {
    global $botToken;
    
    if (!is_numeric($userId) || $userId <= 0 || empty($channel)) {
        return false;
    }
    
    $response = sendTelegramRequest('getChatMember', array(
        'chat_id' => $channel,
        'user_id' => $userId
    ));
    
    if ($response === false) {
        logBotError("Failed to check channel membership for user ID: {$userId}");
        return false;
    }
    
    $data = json_decode($response, true);
    
    if ($data && $data['ok']) {
        $status = $data['result']['status'];
        return in_array($status, array('member', 'administrator', 'creator'));
    }
    
    return false;
}

// تابع نمایش پنل آموزش
function showTutorialPanel($chatId, $messageId = null) {
    global $link3, $link4, $link5;
    
    $safeLink3 = filter_var($link3, FILTER_VALIDATE_URL) ? $link3 : '';
    $safeLink4 = filter_var($link4, FILTER_VALIDATE_URL) ? $link4 : '';
    $safeLink5 = filter_var($link5, FILTER_VALIDATE_URL) ? $link5 : '';
    
    $keyboard = array(
        'inline_keyboard' => array(
            array(
                array('text' => '📱  آخرین نسخه V2rayNG برای آندروید', 'url' => $safeLink3)
            ),
            array(
                array('text' => '🍎  آخرین نسخه V2box برای ios و mac', 'url' => $safeLink4)
            ),
            array(
                array('text' => '🖥️  آخرین نسخه V2rayN برای  ویندوز', 'url' => $safeLink5)
            ),
            array(
                array('text' => '🔙  بازگشت ', 'callback_data' => 'back_to_main')
            )
        )
    );
    
    $messageText = " 🧩  لیست نرم افزارها

آخرین ورژن نرم افزار مربوطه را دانلود و نصب کنید.";
    $messageText = makeTextBold($messageText);
    
    if ($messageId) {
        $result = sendTelegramRequest('editMessageText', array(
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $messageText,
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ));
    } else {
        $result = sendTelegramRequest('sendMessage', array(
            'chat_id' => $chatId,
            'text' => $messageText,
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ));
    }
    
    if ($result === false) {
        logBotError("Failed to show tutorial panel to chat ID: {$chatId}");
    }
}

// تابع نمایش پنل تلگرام پرمیوم
function showTelegramPremiumPanel($chatId, $messageId = null) {
    global $supportUrl;
    
    $safeSupportUrl = filter_var($supportUrl, FILTER_VALIDATE_URL) ? $supportUrl : '';
    
    $keyboard = array(
        'inline_keyboard' => array(
            array(
                array('text' => '👤  پشتیبانی', 'url' => $safeSupportUrl)
            ),
            array(
                array('text' => '🔙  بازگشت ', 'callback_data' => 'back_to_main')
            )
        )
    );
    
    $messageText = "⭐  اشتراک تلگرام پرمیوم بصورت گیفتی و آنی\n\nاشتراک پرمیوم   3  ماهه   1490  تومان\nاشتراک پرمیوم   6  ماهه   1900  تومان\nاشتراک پرمیوم  12  ماهه  3350  تومان\n\n💳  برای خرید به پشتیبانی پیام بدین 👇";
    $messageText = makeTextBold($messageText);
    
    if ($messageId) {
        $result = sendTelegramRequest('editMessageText', array(
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $messageText,
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ));
    } else {
        $result = sendTelegramRequest('sendMessage', array(
            'chat_id' => $chatId,
            'text' => $messageText,
            'reply_markup' => json_encode($keyboard),
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true
        ));
    }
    
    if ($result === false) {
        logBotError("Failed to show telegram premium panel to chat ID: {$chatId}");
    }
}

// بستن اتصال دیتابیس و نوشتن لاگ‌های batch
try {
    $db = null;
    
    // نوشتن لاگ‌های batch
    if (!empty($botErrorLogs)) {
        try {
            secureFilePutContents($botErrorsLog, implode('', $botErrorLogs));
        } catch (Exception $e) {
            file_put_contents($botErrorsLog, implode('', $botErrorLogs), FILE_APPEND | LOCK_EX);
        }
    }
    
    if (!empty($generalErrorLogs)) {
        try {
            secureFilePutContents($errorLog, implode('', $generalErrorLogs));
        } catch (Exception $e) {
            file_put_contents($errorLog, implode('', $generalErrorLogs), FILE_APPEND | LOCK_EX);
        }
    }
} catch (PDOException $e) {
    logGeneralError("Database close error: " . $e->getMessage());
}
?>